<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\User;
use Illuminate\Support\Facades\Gate; // Import Gate facade

class AdminsComponent extends Component
{
    use WithPagination;

    public $perPage = 10;

    public function deleteAdmin($adminId)
    {
        if (!Gate::allows('manage-admins')) {
            session()->flash('error', 'You are not authorized to delete administrators.');
            return;
        }

        $admin = User::find($adminId);
        if ($admin) {
            // Prevent super_admin from deleting themselves or other super_admins
            if ($admin->role === 'super_admin' && auth()->user()->id !== $admin->id) {
                session()->flash('error', 'You cannot delete another Super Admin.');
                return;
            }
            if ($admin->id === auth()->user()->id) {
                session()->flash('error', 'You cannot delete your own account.');
                return;
            }

            $admin->delete();
            session()->flash('message', 'Administrator deleted successfully.');
        } else {
            session()->flash('error', 'Administrator not found.');
        }
    }

    public function updateAdminRole($adminId, $newRole)
    {
        if (!Gate::allows('manage-admins')) {
            session()->flash('error', 'You are not authorized to change administrator roles.');
            return;
        }

        $admin = User::find($adminId);
        if ($admin) {
            // Prevent super_admin from changing their own role or other super_admins to a non-super_admin role
            if ($admin->role === 'super_admin' && $newRole !== 'super_admin' && auth()->user()->id !== $admin->id) {
                session()->flash('error', 'You cannot demote another Super Admin.');
                return;
            }
            if ($admin->id === auth()->user()->id && $newRole !== 'super_admin') {
                session()->flash('error', 'You cannot demote your own account.');
                return;
            }

            $admin->role = $newRole;
            $admin->save();
            session()->flash('message', 'Administrator role updated successfully.');
        } else {
            session()->flash('error', 'Administrator not found.');
        }
    }

    public function render()
    {
        $admins = User::whereIn('role', ['admin', 'super_admin'])
                      ->orderBy('created_at', 'desc')
                      ->paginate($this->perPage);

        return view('livewire.admin.admins-component', compact('admins'))
            ->layout('layouts.adminbase');
    }
}
