<?php

namespace App\Livewire\Admin;


use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Job;
use Illuminate\Support\Facades\Auth;

class AdminPostJobsComponent extends Component
{
    use WithPagination;

    public $job_title, $job_location, $description, $department, $employment_type, $minimum_experience, $salary, $status = 'active';
    public $jobId;
    public $showModal = false;
    public $isEdit = false;
    public $perPage = 10;

    protected $rules = [
        'job_title' => 'required|string|max:255',
        'job_location' => 'nullable|string|max:255',
        'description' => 'required|string',
        'department' => 'nullable|string|max:255',
        'employment_type' => 'nullable|in:full-time,part-time,contract,temporary',
        'minimum_experience' => 'nullable|string|max:255',
        'salary' => 'nullable|string|max:255',
        'status' => 'required|in:active,closed',
    ];

    public function openModal()
    {
        $this->resetForm();
        $this->isEdit = false;
        $this->showModal = true;
        $this->dispatch('show-modal');
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->dispatch('hide-modal');
    }

    public function save()
    {
        $this->validate();

        try {
            if ($this->isEdit) {
                $job = Job::findOrFail($this->jobId);
                $job->update([
                    'title' => $this->job_title,
                    'location' => $this->job_location,
                    'description' => $this->description,
                    'department' => $this->department,
                    'employment_type' => $this->employment_type,
                    'minimum_experience' => $this->minimum_experience,
                    'salary' => $this->salary,
                    'status' => $this->status,
                ]);
            } else {
                Job::create([
                    'title' => $this->job_title,
                    'location' => $this->job_location,
                    'description' => $this->description,
                    'department' => $this->department,
                    'employment_type' => $this->employment_type,
                    'minimum_experience' => $this->minimum_experience,
                    'salary' => $this->salary,
                    'status' => $this->status,
                    'posted_by' => Auth::id(),
                ]);
            }

            $this->closeModal();
            session()->flash('success', 'Job ' . ($this->isEdit ? 'updated' : 'posted') . ' successfully.');
            $this->dispatch('tableUpdated');
        } catch (\Exception $e) {
            \Log::error('Job save failed: ' . $e->getMessage());
            $this->addError('form', 'Failed to save job.');
        }
    }

    public function editJob($jobId)
    {
        $job = Job::findOrFail($jobId);
        $this->jobId = $job->id;
        $this->job_title = $job->title;
        $this->job_location = $job->location;
        $this->description = $job->description;
        $this->department = $job->department;
        $this->employment_type = $job->employment_type;
        $this->minimum_experience = $job->minimum_experience;
        $this->salary = $job->salary;
        $this->status = $job->status;
        $this->isEdit = true;
        $this->showModal = true;
        $this->dispatch('show-modal');
    }

    public function deleteJob($jobId)
    {
        try {
            Job::findOrFail($jobId)->delete();
            session()->flash('success', 'Job deleted successfully.');
        } catch (\Exception $e) {
            \Log::error('Job deletion failed: ' . $e->getMessage());
            session()->flash('error', 'Failed to delete job.');
        }
    }

    public function resetForm()
    {
        $this->reset([
            'jobId', 'job_title', 'job_location', 'description', 'department',
            'employment_type', 'minimum_experience', 'salary', 'status',
        ]);
        $this->resetErrorBag();
    }

    public function render()
    {
        $jobs = Job::with('user')->orderBy('created_at', 'desc')->paginate($this->perPage);
        return view('livewire.admin.admin-post-jobs-component', compact('jobs'))
            ->layout('layouts.adminbase');
    }
}
