<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\Category;
use App\Models\Equipment;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Carbon\Carbon;
use CloudinaryLabs\CloudinaryLaravel\Facades\Cloudinary;
use Illuminate\Support\Facades\Log;

class AdminEquipmentFormComponent extends Component
{
    use WithFileUploads;

    public $name;
    public $slug;
    public $specifications;
    public $status = 'available';
    public $pricing;
    public $description;
    public $image;
    public $category_id;
    public $newimage;
    public $equipment_id;

    public function mount($slug = null)
    {
        if($slug)
        {
            $equipment = Equipment::where('slug', $slug)->first();
            $this->name = $equipment->name;
            $this->slug = $equipment->slug;
            $this->specifications = $equipment->specifications;
            $this->status = $equipment->status;
            $this->pricing = $equipment->pricing;
            $this->description = $equipment->description;
            $this->image = $equipment->image;
            $this->category_id = $equipment->category_id;
            $this->equipment_id = $equipment->id;
        }
    }

    public function generateSlug()
    {
        $this->slug = Str::slug($this->name);
    }

    public function rules()
    {
        return [
            'name' => 'required',
            'slug' => 'required|unique:equipment,slug,'.$this->equipment_id,
            'status' => 'required',
            'pricing' => 'numeric|nullable',
            'image' => 'nullable|image|max:40960',
            'newimage' => 'nullable|image|max:40960',
            'category_id' => 'required'
        ];
    }

    public function updated($fields)
    {
        if($fields == 'image' || $fields == 'newimage') {
            return;
        }
        $this->validateOnly($fields);
    }

    public function addEquipment()
    {
        $this->validate([
            'name' => 'required',
            'slug' => 'required|unique:equipment',
            'status' => 'required',
            'pricing' => 'numeric|nullable',
            'image' => 'required|image|max:40960',
            'category_id' => 'required'
        ]);

        $equipment = new Equipment();
        $equipment->name = $this->name;
        $equipment->slug = $this->slug;
        $equipment->specifications = $this->specifications;
        $equipment->status = $this->status;
        $equipment->pricing = $this->pricing;
        $equipment->description = $this->description;
        
        if ($this->image) {
            try {
                Log::info('Uploading equipment image to Cloudinary: ' . $this->image->getRealPath());
                $uploaded = Cloudinary::upload($this->image->getRealPath(), [
                    'folder' => 'equipment_images',
                    'transformation' => [
                        'quality' => 'auto',
                        'fetch_format' => 'auto'
                    ]
                ]);
                $equipment->image = $uploaded->getSecurePath();
                Log::info('Equipment image uploaded successfully: ' . $equipment->image);
            } catch (\Exception $e) {
                Log::error('Failed to upload equipment image: ' . $e->getMessage());
                session()->flash('error', 'Failed to upload image: ' . $e->getMessage());
                return;
            }
        }

        $equipment->category_id = $this->category_id;
        $equipment->save();
        session()->flash('message', 'Equipment has been created successfully!');
        return redirect()->route('admin.equipment');
    }

    public function updateEquipment()
    {
        $this->validate([
            'name' => 'required',
            'slug' => 'required|unique:equipment,slug,'.$this->equipment_id,
            'status' => 'required',
            'pricing' => 'numeric|nullable',
            'category_id' => 'required',
            'newimage' => 'nullable|image|max:40960'
        ]);

        $equipment = Equipment::find($this->equipment_id);
        $equipment->name = $this->name;
        $equipment->slug = $this->slug;
        $equipment->specifications = $this->specifications;
        $equipment->status = $this->status;
        $equipment->pricing = $this->pricing;
        $equipment->description = $this->description;
        
        if($this->newimage)
        {
            try {
                Log::info('Uploading new equipment image to Cloudinary: ' . $this->newimage->getRealPath());
                $uploaded = Cloudinary::upload($this->newimage->getRealPath(), [
                    'folder' => 'equipment_images',
                    'transformation' => [
                        'quality' => 'auto',
                        'fetch_format' => 'auto'
                    ]
                ]);
                $equipment->image = $uploaded->getSecurePath();
                Log::info('New equipment image uploaded successfully: ' . $equipment->image);
            } catch (\Exception $e) {
                Log::error('Failed to upload new equipment image: ' . $e->getMessage());
                session()->flash('error', 'Failed to upload image: ' . $e->getMessage());
                return;
            }
        }

        $equipment->category_id = $this->category_id;
        $equipment->save();
        session()->flash('message', 'Equipment has been updated successfully!');
        return redirect()->route('admin.equipment');
    }

    public function render()
    {
        $categories = Category::all();
        return view('livewire.admin.admin-equipment-form-component', ['categories' => $categories])->layout('layouts.adminbase');
    }
}
